defmodule Pile do
  @moduledoc """
  Module définissant une pile de cartes.

  On peut ajouter des cartes sur cette pile. Utiliser la fonction
  `ajouter` pour ce faire : la carte ajoutée sera placée au sommet
  de la pile.
  """

  defstruct cartes: []

  @typedoc "Une pile de cartes dans un ordre précis."
  @type t() :: %Pile{cartes: [Carte.t()]}

  @doc """
  Crée une pile vide.

  ## Exemples

      iex> Pile.new()
      %Pile{cartes: []}

  """
  @spec new() :: t()
  def new(), do: %Pile{}

  @doc """
  Ajoute une carte à la pile.

  Pour ajouter une carte, préciser la pile, ainsi que la valeur
  et l'enseigne de la carte à ajouter. Si la carte a été ajoutée,
  retourne la pile contenant cette nouvelle carte. Sinon, retourne
  l'erreur sous forme d'atome.

  ## Exemples

      iex> Pile.new |> Pile.ajouter("5", "trèfle")
      %Pile{cartes: [%Carte{enseigne: "trèfle", valeur: "5"}]}
      iex> Pile.new |> Pile.ajouter("sept", "trèfle")
      :valeur_invalide
      iex> Pile.new |> Pile.ajouter("7", "autre")
      :enseigne_invalide

  """
  @spec ajouter(t(), String.t(), String.t()) :: t() | Carte.carte_invalide()
  def ajouter(pile, valeur, enseigne) do
    carte = Carte.new(valeur, enseigne)

    case carte do
      %Carte{} ->
        %Pile{cartes: [carte | pile.cartes]}

      erreur ->
        erreur
    end
  end
end
